import adafruit_logging as logging
log = logging.getLogger()
import time

from core.game_state import GameState, STATE_ADMIN, STATE_PLAYING, STATE_PAUSE, STATE_FINISHED, STATE_START
from core.view.view import View
from microcontroller import Pin
from catan.player_mapper import Player_Mapper
from pwmio import PWMOut

SOLID_ON_DUTY_CYCLE = 65535

# A view of a number of PWM-enabled standard LEDs, one for each seat.
# The pins should be the IO pins for the LEDs.
# The list of pins should be in seat-order, so pin index 0 is for seat #1 etc.
class PWM_Leds_View(View):
	leds: dict[Pin, PWMOut]
	player_mapper: Player_Mapper
	turn_time_sec: int|None

	def __init__(self, player_mapper: Player_Mapper):
		super().__init__()
		self.player_mapper = player_mapper
		self.leds = {pin: PWMOut(pin, frequency=5000, duty_cycle=0) for pin in player_mapper.all_led_pins}
		self.turn_time_sec = None


	# See inline comments for info on the different animations of the LEDs.
	def animate(self) -> bool:
		# This is the PWM duty cycle of whatever LEDs that should be active,
		# either one (active_led) or all of them
		# By default, if we have no game state, we have a slow pulse.
		duty_cycle = self.seconds_per_blink(1, blinks_before_pause=1, pause_duration=3)

		if self.state != None:
			if self.state.state == STATE_PLAYING:
				timings = self.state.get_current_timings()
				turn_time = timings.turn_time
				duty_cycle = SOLID_ON_DUTY_CYCLE if turn_time < 0 else self.seconds_per_blink(2, pause_duration=0.2, inverse=True)
			elif self.state.state == STATE_ADMIN:
				duty_cycle = self.seconds_per_blink(2, blinks_before_pause=1, pause_duration=2, min=0.04, max=0.4)
			elif self.state.state == STATE_PAUSE:
				duty_cycle = self.seconds_per_blink(1, blinks_before_pause=2, pause_duration=2)
			elif self.state.state not in [STATE_FINISHED, STATE_START]:
				# Not sure what this state would be...
				duty_cycle = self.seconds_per_blink(0.5)

		active_led_pins = self.player_mapper.active_led_pins
		for led_pin in self.leds:
			pwm = self.leds[led_pin]
			if led_pin in active_led_pins:
				pwm.duty_cycle = duty_cycle
			else:
				pwm.duty_cycle = 0

		return False # This view is never busy.

	def on_state_update(self, state: GameState|None, old_state: GameState|None):
		print(state)
		self.player_mapper.handle_new_state(state)

	def seconds_per_blink(self, single_blink_duration: float, min = 0.0, max = 1.0, blinks_before_pause = 1, pause_duration = 0, inverse=False):
		result = min * SOLID_ON_DUTY_CYCLE
		max_add = max * SOLID_ON_DUTY_CYCLE - result
		blinking_duration = single_blink_duration * blinks_before_pause
		cycle_duration = blinking_duration + pause_duration
		_cycle_count, seconds_into_cycle = divmod(time.monotonic(), cycle_duration)
		if seconds_into_cycle < blinking_duration:
			divider = single_blink_duration / 2
			quotient, remainder = divmod(seconds_into_cycle, divider)
			if (quotient % 2 == 0):
				result += (remainder / divider) * max_add # Up
			else:
				result += ((divider-remainder) / divider) * max_add # Down
		result = int(result)
		return SOLID_ON_DUTY_CYCLE - result if inverse else result
