import adafruit_logging as logging
log = logging.getLogger()
import time
import json
from core.view.view import View
from core.connection.sgt_connection_stream import SgtConnectionStream
from adafruit_ble import BLERadio
from adafruit_ble.advertising.standard import ProvideServicesAdvertisement
from adafruit_ble.services.nordic import UARTService

class SgtConnectionBluetooth(SgtConnectionStream):
	def __init__(self,
				view: View,
				device_name: str,
				field_order: list[str],
				field_divider: str,
				):
		super().__init__(view, field_order, field_divider)
		self.ble = BLERadio()
		self.uart = UARTService()
		self.advertisement = ProvideServicesAdvertisement(self.uart)
		self.ble.name = device_name
		self.last_is_connected_check = False
		self.byte_array = bytearray(20)

	def is_connected(self) -> bool:
		if self.ble.connected and not self.last_is_connected_check:
			self.view.set_connection_progress_text('Establishing Connection')
			self.ble.stop_advertising()
			# Wait for the first poll request to go through,
			time_of_last_poll_request = 0
			while self.ble.connected and self.uart.in_waiting == 0:
				if time.monotonic() - time_of_last_poll_request > 0.5:
					time_of_last_poll_request = time.monotonic()
					log.info('Waiting for ping')
					self.send('Ping')
				self.view.animate()
			if self.ble.connected:
				log.info('Ping Acknowledged')
				self.send('Enable ACK')
				self.uart.reset_input_buffer()
				self.send('Poll')
			else:
				raise Exception('Disconnected while waiting for ping')

		self.last_is_connected_check = self.ble.connected
		return self.ble.connected

	def connect(self):
		self.view.set_connection_progress_text(f"Advertising BLE as {self.ble.name}")
		self.ble.start_advertising(self.advertisement)

	def read_text_from_stream(self) -> str | None:
		if self.uart.in_waiting == 0:
			if self.incomplete_line_read and time.monotonic() - self.incomplete_line_read[0] > 6:
				log.debug('Old incomplete line. Clear the buffer and line, then call poll for new data. %s', self.incomplete_line_read)
				self.uart.reset_input_buffer()
				self.incomplete_line_read = None
				self._poll_for_latest_state()
			return None
		else:
			bytes_read = self.uart.readinto(buf=self.byte_array, nbytes=self.uart.in_waiting)
			result = str(self.byte_array[:bytes_read], 'utf-8')
			self.send('ACK')
			return result

	def pre_process_line(self, line):
		if line[1] == 'GET SETUP':
			log.info('SENDING SUGGESTED SETUP')
			suggestions = json.dumps({
			"script": [
				f'0 %0A{self.field_divider.join(self.field_order)}%0A'
			],
				"scriptName": self.ble.name + " Write",
				"defaultTriggers": ["includePlayers","includePause","includeAdmin","includeSimultaneousTurns","includeGameStart","includeGameEnd","includeSandTimerStart","includeSandTimerReset","includeSandTimerStop","includeSandTimerOutOfTime","runOnStateChange","runOnPlayerOrderChange", "runOnPoll"],
				"actionMap": [
					('Enable ACK', 'remoteActionEnableAcknowledge'),
					('ACK', 'remoteActionAcknowledge'),
					('Poll', 'remoteActionPoll'),
					('Primary', 'remoteActionPrimary'),
					('Secondary', 'remoteActionSecondary'),
					('Undo', 'remoteActionUndo'),
					('ToggleAdmin', 'remoteActionToggleAdmin'),
					('TurnAdminOn', 'remoteActionTurnAdminOn'),
					('TurnAdminOff', 'remoteActionTurnAdminOff'),
					('TogglePause', 'remoteActionTogglePause'),
					('TurnPauseOn', 'remoteActionTurnPauseOn'),
					('TurnPauseOff', 'remoteActionTurnPauseOff'),
				],
				"actionMapName": "Hardcoded Actions",
			})
			self.send(suggestions)
			return None
		else:
			return line

	def send(self, value: str|None):
		if value == None:
			return
		self.uart.write((value+"\n").encode("utf-8"))

	def _poll_for_latest_state(self):
		log.info('Polling for new data')
		self.send("Poll")