import adafruit_logging as logging
log = logging.getLogger()
import time
from core.view.view import View
from core.connection.sgt_connection import SgtConnection
from core.game_state import GameState

class SgtConnectionStream(SgtConnection):
	def __init__(self,
				view: View,
				field_order: list[str],
				field_divider: str,
				):
		super().__init__(view)
		self.incomplete_line_read = None
		self.command_to_send = None
		self.line_to_process = None
		self.field_order = field_order
		self.field_divider = field_divider

	def read_text_from_stream(self) -> str | None:
		raise NotImplementedError()

	def pre_process_line(self, line: str):
		raise NotImplementedError()

	def send(self, value: str|None):
		raise NotImplementedError()

	def poll_for_new_messages(self) -> None:
		while True:
			read_text = self.read_text_from_stream()
			if read_text == None:
				return
			lines = [(time.monotonic(), line) for line in read_text.split("\n")]
			if self.incomplete_line_read != None:
				lines[0] = (self.incomplete_line_read[0], self.incomplete_line_read[1]+lines[0][1])
				self.incomplete_line_read = None

			if len(lines) == 0:
				return

			# Keep hold of the very last item. We need to know if it is an incomplete line or
			last_item = lines.pop()
			# Filter out empty lines
			lines = [line for line in lines if line[1] != '']
			if len(lines[:-1]) > 0:
				log.debug(f'SKIP: {lines[:-1]}')

			if last_item[1] == '':
				# We got a new line. So, we should execute the line before it...
				if len(lines) > 0:
					# ...if there is one.
					self.line_to_process = self.pre_process_line(lines.pop())
			else:
				# We are not done reading the next line.
				self.incomplete_line_read = last_item
				time.sleep(0.05)
	def handle_new_messages(self) -> None:
		if self.line_to_process == None:
			return False
		log.info(f"<-- {self.line_to_process}")
		line_time, line_state_string = self.line_to_process
		self.line_to_process = None

		new_state = GameState(
			ble_state_string = line_state_string,
			ble_field_order = self.field_order,
			ble_field_divider = self.field_divider,
			timestamp_offset = line_time - time.monotonic()
			)
		self.view.set_state(new_state)
		return True

	def _enqueue_command(self, value: str):
		if value != None:
			self.command_to_send = value

	def send_command(self) -> bool:
		if self.command_to_send == None:
			return False
		else:
			self.send(self.command_to_send)
			self.command_to_send = None
			return True

	def enqueue_send_primary(self, seat: int|None = None, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		action = super().enqueue_send_primary(seat, on_success, on_failure)
		if action != None and seat != None:
			self._enqueue_command(f'{action} #{seat}')
		else:
			self._enqueue_command(action)
	def enqueue_send_secondary(self, seat: int|None = None, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		action = super().enqueue_send_secondary(seat, on_success, on_failure)
		if action != None and seat != None:
			self._enqueue_command(f'{action} #{seat}')
		else:
			self._enqueue_command(action)
	def enqueue_send_toggle_admin(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_toggle_admin(on_success, on_failure))
	def enqueue_send_admin_on(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_admin_on(on_success, on_failure))
	def enqueue_send_admin_off(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_admin_off(on_success, on_failure))
	def enqueue_send_toggle_pause(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_toggle_pause(on_success, on_failure))
	def enqueue_send_pause_on(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_pause_on(on_success, on_failure))
	def enqueue_send_pause_off(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_pause_off(on_success, on_failure))
	def enqueue_send_undo(self, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		self._enqueue_command(super().enqueue_send_undo(on_success, on_failure))
	def enqueue_send_start_game(self, seat: int|None = None, seats: list[int]|None = None, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		command = super().enqueue_send_start_game(seat, seats, on_success, on_failure)
		if command == None:
			return False
		if seat != None:
			self._enqueue_command(f'{command} #{seat}')
		elif seats != None:
			self._enqueue_command(f'{command} #{",".join([str(s) for s in seats])}')
		else:
			self._enqueue_command(command)
	def enqueue_send_start_sim_turn(self, seats: set[int]):
		command = super().enqueue_send_start_sim_turn(seats)
		if command != None:
			self._enqueue_command(f'{command} #{",".join([str(s) for s in seats])}')
			return True
		else:
			return False
	def enqueue_send_join_game_or_cycle_colors(self, seat: int, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		action = super().enqueue_send_join_game_or_cycle_colors(seat, on_success, on_failure)
		if action != None and seat != None:
			self._enqueue_command(f'{action} #{seat}')
	def enqueue_send_leave_game(self, seat: int, on_success: callable[[], None] = None, on_failure: callable[[], None] = None):
		action = super().enqueue_send_leave_game(seat, on_success, on_failure)
		if action != None and seat != None:
			self._enqueue_command(f'{action} #{seat}')
